﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using CashierCrisis.Core;

namespace CashierCrisis.Graphs
{
    public class AStarGraphSearch<N,E>
        where N: GraphNode
        where E: GraphEdge

    {
        private SparseGraph<N, E> _graph;


        private List<double> _gCosts;
        private List<double> _fCosts;
        private List<E> _shortestPathTree;
        private List<E> _searchFrontier;
        private int _source;
        private int _target;
        private  IHeuristic<SparseGraph<N,E>,N,E> _heuristic;

        public AStarGraphSearch(SparseGraph<N,E> graph, int source, int target, IHeuristic<SparseGraph<N,E>,N,E> heuristic)
        {
            _graph = graph;
            _source = source;
            _target = target;

            _shortestPathTree = new List<E>(graph.NodeCount);
            _searchFrontier = new List<E>(graph.NodeCount);
            _fCosts = new List<double>(graph.NodeCount);
            _gCosts = new List<double>(graph.NodeCount);

            for (int i = 0; i < graph.NodeCount; i++)
            {
                _shortestPathTree.Add(null);
                _searchFrontier.Add(null);
                _fCosts.Add(0);
                _gCosts.Add(0);
            }

            _heuristic = heuristic;
            Search();
        }

        

        public List<E> ShortestPathTree
        {
            get { return _shortestPathTree; }
        }

        public List<int> PathToTarget()
        {
            List<int> path = new List<int>();

            if (_target < 0)
                return path;

            int n = _target;

            path.Add(n);


           while((n != _source) && (_shortestPathTree[n] !=null))
           {
               n = _shortestPathTree[n].From;
               path.Add(n);
           }

            path.Reverse();
            return path;
        }

        public Queue<int> PathToTargetQ()
        {
            Queue<int> path = new Queue<int>();

            if (_target < 0)
                return path;
            int n = _target;
            path.Enqueue(n);

            while((n != _source) && (_shortestPathTree[n] !=null))
            {
                n = _shortestPathTree[n].From;
                path.Enqueue(n);
            }
            return path;
        }

        public List<int> GetPathToTarget(int source, int target)
        {
            _source = source;
            _target = target;
            Search();
            return PathToTarget();
        }

        public Queue<int> GetPathToTargetQ(int source, int target)
        {
            _source = source;
            _target = target;
            Search();
            return PathToTargetQ();
        }

        public double CostToTarget
        {
            get
            {
                return _gCosts[_target];
            }
            
        }

        public void Search()
        {
            IndexedPriorityQueueLow<double> pq = new IndexedPriorityQueueLow<double>(_fCosts, _graph.NodeCount);
            
            pq.Insert(_source);

            while (!pq.Empty)
            {
                int NextClosestNode = pq.Pop();

                _shortestPathTree[NextClosestNode] = _searchFrontier[NextClosestNode];

                if (NextClosestNode == _target)
                    return;

               foreach(E edge in _graph.Edges(NextClosestNode))
               {
                   double hCost = _heuristic.Calculate(_graph, _target, _source);

                   double gCost = _gCosts[NextClosestNode] + edge.To;

                   if (_searchFrontier[edge.To] == null)
                   {
                       _fCosts[edge.To] = gCost + hCost;
                       _gCosts[edge.To] = gCost;

                       pq.Insert(edge.To);
                       _searchFrontier[edge.To] = edge;
                   }

                   else if (gCost < _gCosts[edge.To] &&
                       _shortestPathTree[edge.To] == null)
                   {
                       _fCosts[edge.To] = gCost + hCost;
                       _gCosts[edge.To] = gCost;
                       pq.ChangePriority(edge.To);
                       _searchFrontier[edge.To] = edge;
                   }
               }
               
            }
        }
    }
}
